package de.flapdoodle.embed.mongo.spring.autoconfigure;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link MongoClientAction.Credentials}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableCredentials.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableCredentials
    extends MongoClientAction.Credentials {
  private final String database;
  private final String username;
  private final String password;

  private ImmutableCredentials(String database, String username, String password) {
    this.database = database;
    this.username = username;
    this.password = password;
  }

  /**
   * @return The value of the {@code database} attribute
   */
  @Override
  public String database() {
    return database;
  }

  /**
   * @return The value of the {@code username} attribute
   */
  @Override
  public String username() {
    return username;
  }

  /**
   * @return The value of the {@code password} attribute
   */
  @Override
  public String password() {
    return password;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link MongoClientAction.Credentials#database() database} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for database
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableCredentials withDatabase(String value) {
    String newValue = Objects.requireNonNull(value, "database");
    if (this.database.equals(newValue)) return this;
    return new ImmutableCredentials(newValue, this.username, this.password);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link MongoClientAction.Credentials#username() username} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for username
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableCredentials withUsername(String value) {
    String newValue = Objects.requireNonNull(value, "username");
    if (this.username.equals(newValue)) return this;
    return new ImmutableCredentials(this.database, newValue, this.password);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link MongoClientAction.Credentials#password() password} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for password
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableCredentials withPassword(String value) {
    String newValue = Objects.requireNonNull(value, "password");
    if (this.password.equals(newValue)) return this;
    return new ImmutableCredentials(this.database, this.username, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableCredentials} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableCredentials
        && equalTo(0, (ImmutableCredentials) another);
  }

  private boolean equalTo(int synthetic, ImmutableCredentials another) {
    return database.equals(another.database)
        && username.equals(another.username)
        && password.equals(another.password);
  }

  /**
   * Computes a hash code from attributes: {@code database}, {@code username}, {@code password}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + database.hashCode();
    h += (h << 5) + username.hashCode();
    h += (h << 5) + password.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Credentials} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Credentials{"
        + "database=" + database
        + ", username=" + username
        + ", password=" + password
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link MongoClientAction.Credentials} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Credentials instance
   */
  public static ImmutableCredentials copyOf(MongoClientAction.Credentials instance) {
    if (instance instanceof ImmutableCredentials) {
      return (ImmutableCredentials) instance;
    }
    return ImmutableCredentials.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableCredentials ImmutableCredentials}.
   * <pre>
   * ImmutableCredentials.builder()
   *    .database(String) // required {@link MongoClientAction.Credentials#database() database}
   *    .username(String) // required {@link MongoClientAction.Credentials#username() username}
   *    .password(String) // required {@link MongoClientAction.Credentials#password() password}
   *    .build();
   * </pre>
   * @return A new ImmutableCredentials builder
   */
  public static ImmutableCredentials.Builder builder() {
    return new ImmutableCredentials.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableCredentials ImmutableCredentials}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_DATABASE = 0x1L;
    private static final long INIT_BIT_USERNAME = 0x2L;
    private static final long INIT_BIT_PASSWORD = 0x4L;
    private long initBits = 0x7L;

    private String database;
    private String username;
    private String password;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Credentials} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(MongoClientAction.Credentials instance) {
      Objects.requireNonNull(instance, "instance");
      database(instance.database());
      username(instance.username());
      password(instance.password());
      return this;
    }

    /**
     * Initializes the value for the {@link MongoClientAction.Credentials#database() database} attribute.
     * @param database The value for database 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder database(String database) {
      this.database = Objects.requireNonNull(database, "database");
      initBits &= ~INIT_BIT_DATABASE;
      return this;
    }

    /**
     * Initializes the value for the {@link MongoClientAction.Credentials#username() username} attribute.
     * @param username The value for username 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder username(String username) {
      this.username = Objects.requireNonNull(username, "username");
      initBits &= ~INIT_BIT_USERNAME;
      return this;
    }

    /**
     * Initializes the value for the {@link MongoClientAction.Credentials#password() password} attribute.
     * @param password The value for password 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder password(String password) {
      this.password = Objects.requireNonNull(password, "password");
      initBits &= ~INIT_BIT_PASSWORD;
      return this;
    }

    /**
     * Builds a new {@link ImmutableCredentials ImmutableCredentials}.
     * @return An immutable instance of Credentials
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableCredentials build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableCredentials(database, username, password);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_DATABASE) != 0) attributes.add("database");
      if ((initBits & INIT_BIT_USERNAME) != 0) attributes.add("username");
      if ((initBits & INIT_BIT_PASSWORD) != 0) attributes.add("password");
      return "Cannot build Credentials, some of required attributes are not set " + attributes;
    }
  }
}
